#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "oled_driver.h"
#include "oled_font.h"

#warning "If type bool is not available, define bool as this, comment them if you got bool type"
#define false 0x00
#define true  (!false)
#define PI 3.14159265358979323846f

static uint8_t RAM[OLED_PAGE][OLED_WIDTH] = {0x00};
static uint8_t OLED_CMD[16] = {0x00};

#warning "OLED_IIC_Write() is not implemented"
__weak uint8_t OLED_IIC_Write(uint8_t dc, uint8_t* payload, uint16_t size){}

#warning "OLED_Reset() is not implemented"
__weak void OLED_Reset(){}

#warning "OLED_Init() should be re-implemented according to specific OLED panel"
__weak uint8_t OLED_Init(){
  OLED_Reset();
  if(OLED_Display(false)) return 0x01;
  if(OLED_ConfigClock(0x0F,0x00)) return 0x02;
  if(OLED_MuxRatio(0x3F)) return 0x03;
  if(OLED_OffsetPanel(0x00)) return 0x04;
  if(OLED_OffsetGRAM(0x00)) return 0x05;
  if(OLED_ReverseHorizontal(true)) return 0x06;
  if(OLED_ReverseVertical(true)) return 0x07;
  if(OLED_ConfigCommon(false,false,OLED_VCOMH_083)) return 0x08;
  if(OLED_Contrast(0xFF)) return 0x09;
  if(OLED_ChargePhase(0x0F,0x01)) return 0x0A;
  if(OLED_ChargePump(true)) return 0x0B;
  if(OLED_EnableAllPanel(false)) return 0x0C;
  if(OLED_ReverseColor(false)) return 0x0D;
  if(OLED_ScrollCancel()) return 0x0E;
  if(OLED_Display(true)) return 0x0F;
  return 0x00;
}

/**
 * @brief This function update all 1024Byte GRAM data to GDDRAM through single IIC transmission
 * Actually, user can switch to Vertical Addressing mode either, but page addressing is not recomended
 * cause to transmit all data page addressing mode will go for 8 transmission at least
 * @return Whether transmission succeeded
 */
uint8_t OLED_PushAll(){
  if(OLED_AddressHorizontal(0, OLED_WIDTH-1, 0, OLED_PAGE-1)) return 0x01;
  return OLED_IIC_Write(true, &RAM[0][0], OLED_PAGE*OLED_WIDTH);
}

/**
 * @brief This function only push part of the GRAM to GDDRAM and must config addressing params firstly
 * @param page Which page to start the push process
 * @param col  Which column to start the push process
 * @param length How many bytes of data need to push
 * @return Whether the push process is successful
 */
uint8_t OLED_Push(uint8_t page, uint8_t col, uint16_t length){
  if(page>=OLED_PAGE||col>=OLED_WIDTH||length==0) return 0xFF;
  if(page*OLED_WIDTH+col+length>OLED_PAGE*OLED_WIDTH)
    length = OLED_PAGE*OLED_WIDTH-page*OLED_WIDTH+col;
  if(length==0) return 0xFF;
  return OLED_IIC_Write(true, &RAM[page][col], length);
}

/**
 * @brief Fill the whole scree with a single data for a (Seg,Page)
 * @param data 0x00 means clear screen, 0xFF means fill screen, other values are infrequently used
 * @attention All the function below is just operating GRAM but not update to OLED
 */
void OLED_FillAll(uint8_t data){
  memset(RAM, data, OLED_PAGE*OLED_WIDTH);
}

/**
 * @brief Set a single pixel in GRAM as on or off
 * @param x Column(Segment) index
 * @param y Row(Common, not page) index
 * @param on Boolean value, true means light up, false means turning off
 */
void OLED_Point(uint8_t x, uint8_t y, uint8_t on){
  if(x>=OLED_WIDTH||y>=OLED_HEIGHT) return;
  if(on) RAM[y/8][x] |= 1<<(y%8);
  else RAM[y/8][x] &= ~(1<<(y%8));
}

/**
 * @brief Draw a arbitrarily defined with two endpoints to GRAM on or off
 * @param x1 Endpoint 1 column index
 * @param y1 Endpoint 1 row index
 * @param x2 Endpoint 2 column index
 * @param y2 Endpoint 2 row index
 * @param on Boolean value, true means light up, false means turning off
 */
void OLED_Line(uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2, uint8_t on){
  if(x1>=OLED_WIDTH||y1>=OLED_HEIGHT||x2>=OLED_WIDTH||y2>=OLED_HEIGHT) return;
  int16_t dx = (int16_t)(x2 - x1);
  int16_t dy = (int16_t)(y2 - y1);
  int8_t sx = (int8_t)((dx>0)?1:(dx<0)?-1:0);
  int8_t sy = (int8_t)((dy>0)?1:(dy<0)?-1:0);
  dx = (int16_t)abs(dx);
  dy = (int16_t)abs(dy);
  int16_t err = (int16_t)(dx - dy);
  int16_t err2;

  while(1){
    OLED_Point(x1,y1,on);
    if(x1==x2&&y1==y2) break;
    err2 = (int16_t)(err*2);
    if(err2 > -dy) { err=(int16_t)(err-dy); x1 += sx;}
    if(err2 < dx)  { err=(int16_t)(err-dx); y1 += sy;}
  }
}

/**
 * @brief Draw a path defined by a series of nodes to GRAM on or off
 * @param pathX Pointer(array) to the column index of nodes
 * @param pathY Pointer(array) to the row index of nodes
 * @param nodes Nodes count
 * @param on Boolean value, true means light up, false means turning off
 * @param close Boolean value, whether connect the tail to the head node
 */
void OLED_Path(uint8_t* pathX, uint8_t* pathY, uint16_t nodes, uint8_t on, uint8_t close){
  for(uint16_t i=0;i<nodes-1;i++){
    OLED_Line(pathX[i],pathY[i],pathX[i+1],pathY[i+1],on);
  }
  if(close&&nodes>2) OLED_Line(pathX[nodes-1],pathY[nodes-1],pathX[0],pathY[0],on);
}

/**
 * @brief Draw a rectangle defined by several necessary param to GRAM on or off
 * @param cx Central point column index
 * @param cy Central point row index
 * @param angle Rotate angle in degree, horizontal is 0 reference, cw/ccw depends on screen orientation
 * @param width Horizontal scale in pixel(reference angle)
 * @param height Vertical scale in pixel(reference angle)
 * @param on Boolean value, true means light up, false means turning off
 */
void OLED_Rectangle(uint8_t cx, uint8_t cy, float angle, uint8_t width, uint8_t height, uint8_t on){
  if(width==0||height==0) return;
  float sinA = sinf(angle*PI/180.0f);
  float cosA = cosf(angle*PI/180.0f);
  float dx = (float)width/2.0f;
  float dy = (float)height/2.0f;
  float x1 = (float)cx + dx*cosA - dy*sinA;
  float y1 = (float)cy + dx*sinA + dy*cosA;
  float x2 = (float)cx - dx*cosA - dy*sinA;
  float y2 = (float)cy - dx*sinA + dy*cosA;
  float x3 = (float)cx - dx*cosA + dy*sinA;
  float y3 = (float)cy - dx*sinA - dy*cosA;
  float x4 = (float)cx + dx*cosA + dy*sinA;
  float y4 = (float)cy + dx*sinA - dy*cosA;
  OLED_Line((uint8_t)x1,(uint8_t)y1,(uint8_t)x2,(uint8_t)y2,on);
  OLED_Line((uint8_t)x2,(uint8_t)y2,(uint8_t)x3,(uint8_t)y3,on);
  OLED_Line((uint8_t)x3,(uint8_t)y3,(uint8_t)x4,(uint8_t)y4,on);
  OLED_Line((uint8_t)x4,(uint8_t)y4,(uint8_t)x1,(uint8_t)y1,on);
}

/**
 * @brief Draw a regular polygon defined by several necessary param to GRAM on or off.
 * This function is a special case of OLED_Path, which is a regular polygon with all edges equal and
 * the first node will be located at the horizontal reference line.
 * @param cx Central point column index
 * @param cy Central point row index
 * @param radius Radius of external circle in pixel
 * @param poly How many edges the polygon has
 * @param angle Rotate angle in degree, horizontal is 0 reference, cw/ccw depends on screen orientation
 * @param on Boolean value, true means light up, false means turning off
 */
void OLED_RegularPoly(uint8_t cx, uint8_t cy, uint8_t radius, uint8_t poly, uint8_t angle, uint8_t on){
  if(poly<3||radius==0) return;
  uint8_t* xPath = (uint8_t*)malloc(poly*sizeof(uint8_t));
  uint8_t* yPath = (uint8_t*)malloc(poly*sizeof(uint8_t));
  for(uint8_t i=0;i<poly;i++){
    xPath[i] = cx+(uint8_t)((float)radius*cosf((float)i*2.0f*PI/(float)poly+(float)angle*PI/180.0f));
    yPath[i] = cy+(uint8_t)((float)radius*sinf((float)i*2.0f*PI/(float)poly+(float)angle*PI/180.0f));
  }
  OLED_Path(xPath,yPath,poly,on,true);
  free(xPath);
  free(yPath);
}

/**
 * @brief Draw a circular arc defined by several necessary param to GRAM on or off.
 * @param cx Central point column index
 * @param cy Central point row index
 * @param radius Radius of the arc
 * @param startAngle Arc start angle in degrees
 * @param endAngle  Arc end angle in degrees
 * @param on Boolean value, true means light up, false means turning off
 * @Attention Superior arc or inferior arc depends on start/end angle sequence, and if user
 * want to draw a full circle, the start angle angle end angle must be 0 for float calculation error.
 */
void OLED_Arc(uint8_t cx, uint8_t cy, uint8_t radius, float startAngle, float endAngle, uint8_t on){
  float startRad = startAngle*PI/180.0f;
  float endRad = endAngle*PI/180.0f;
  while(startRad>PI) startRad -= 2.0f*PI;
  while(startRad<-PI) startRad += 2.0f*PI;
  while(endRad>PI) endRad -= 2.0f*PI;
  while(endRad<-PI) endRad += 2.0f*PI;
  if(startRad>=endRad) endRad += 2.0f*PI;
  float pace = asinf(1.0f/(float)radius);
  for(float angle=startRad;angle<=endRad;) {
    OLED_Point(cx + (uint8_t) ((float) radius * cosf(angle)),
               cy + (uint8_t) ((float) radius * sinf(angle)), on);
    angle+=pace;
  }
}

/**
 * @brief Draw a circle defined by several necessary param to GRAM on or off.
 * This function is a special case of OLED_Arc, which is a full circle with start angle 0 and end angle 0.
 * @param cx Central point column index
 * @param cy Central point row index
 * @param radius Radius of the circle
 * @param on Boolean value, true means light up, false means turning off
 */
void OLED_Circle(uint8_t cx, uint8_t cy, uint8_t radius, uint8_t on){
  OLED_Arc(cx,cy,radius,0.0f,0.0f,on);
}

/**
 * @brief This function put visible ASCII character to GRAM
 * @param x Start pixel (left-top) column index
 * @param y Start pixel (left-top) row index
 * @param c Target ASCII character, must between 0x20(' ') and 0x7E('~')
 * @param size Font size, must be corresponded to the font library
 * @param on Boolean value, true means light up, false means turning off
 */
void OLED_CharASCII(uint8_t x, uint8_t y, char c, uint8_t size, uint8_t on){
  //Check font size
  if(size!=CHAR_SIZE_12&&size!=CHAR_SIZE_16&&size!=CHAR_SIZE_24) return;
  //Check character visibility
  if(c<0x20||c>0x7E) return;
  uint8_t cIndex = c - 0x20;
  //Calculate how many bytes the charater image costs
  uint8_t bytes = (size/8+((size%8)?1:0))*(size/2);
  uint8_t base = y;
  uint8_t tmp,i,j;
  for(i=0;i<bytes;i++){
    //pick up correspond bytes from font library to tmp
    switch(size){
      case CHAR_SIZE_12: tmp=fontASCII_1206[cIndex][i]; break;
      case CHAR_SIZE_16: tmp=fontASCII_1608[cIndex][i]; break;
      case CHAR_SIZE_24: tmp=fontASCII_2412[cIndex][i]; break;
      default: return;
    }
    //Decode the single bit from the byte(tmp) and write to GRAM
    for(j=0;j<8;j++){
      if(tmp&0x80) OLED_Point(x,y,on);
      else OLED_Point(x,y,!on);
      //Offset and update coordinates
      tmp <<= 1;
      y++;
      if((y-base)==size){y=base;x++;break;}
    }
  }
}

/**
 * @brief This function encapsulate the OLED_CharASCII function and it can put ASCII string to GRAM
 * @param x Start pixel (left-top) column index
 * @param y Start pixel (left-top) row index
 * @param s String pointer or array name which need to put in GRAM
 * @param size Font size, should be corresponded to font library
 * @param Boolean value, true means light up, false means turning off
 */
void OLED_StringASCII(uint8_t x,uint8_t y,const char* s,uint8_t size,uint8_t on){
  char *p = (char*)s;
  while((*p<=0x7E)&&(*p>=0x20)){
    OLED_CharASCII(x,y,*p,size,on);
    x += size/2;
    if(x>=OLED_WIDTH) break;
    p++;
  }
}

/**
 * @brief 0xAE/0xAF command
 * @param on Boolean value, open or close panel display
 * @return Whether transmission succeeded
 */
uint8_t OLED_Display(uint8_t on){
  OLED_CMD[0] = on ? OLED_DISPLAY_ON : OLED_DISPLAY_OFF;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief 0xA6/0xA7 command
 * @param reverse Boolean value, inverse(0=Light) or normal(0=Dark) mode
 * @return Whether transmission succeeded
 */
uint8_t OLED_ReverseColor(uint8_t reverse){
  OLED_CMD[0] = reverse ? OLED_COLOR_INVERSE : OLED_COLOR_NORMAL;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief 0xA4/0xA5 command
 * @param all Boolean value, light entire panel or load from GDDRAM
 * @return Whether transmission succeeded
 */
uint8_t OLED_EnableAllPanel(uint8_t all){
  OLED_CMD[0] = all ? OLED_DATA_ALL : OLED_DATA_RAM;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief 0x81 command, set contrast(brightness)
 * @param contrast 0x00~0xFF value means constrast ratio as (contrast+1)/256
 * @return Whether transmission succeeded
 */
uint8_t OLED_Contrast(uint8_t contrast){
  OLED_CMD[0] = OLED_CONTRAST;
  OLED_CMD[1] = contrast;
  return OLED_IIC_Write(false, OLED_CMD, 2);
}

/**
 * @brief 0xA0/0xA1 command
 * @param reverse Boolean value, reverse=1: display from left to right(GRAM),
 * @return Whether transmission succeeded
 */
uint8_t OLED_ReverseHorizontal(uint8_t reverse){
  OLED_CMD[0] = reverse ? OLED_HORIZONTAL_RL : OLED_HORIZONTAL_LR;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief 0xC0/0xC8 command
 * @param reverse Boolean value, reverse=1: display from top to bottom(GRAM),
 * @return Whether transmission succeeded
 */
uint8_t OLED_ReverseVertical(uint8_t reverse){
  OLED_CMD[0] = reverse ? OLED_VERTICAL_DU : OLED_VERTICAL_UD;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief This function is a composition of horizontal/vertical addressing mode settings.
 * This function is not exposed, 2 more function will re-encapsulated based on this function.
 * @param horizontal Boolean value, horizontal addressing mode or vertical addresing mode
 * @param startCol Start addressing columns index
 * @param endCol End addressing columns index
 * @param startPage Start addressing page index
 * @param endPage End addressing page index
 * @return 0x00 means successful, 0xFF means col/page range out of index, other means failure step
 */
uint8_t OLED_AddressSequence(uint8_t horizontal,uint8_t startCol, uint8_t endCol, uint8_t startPage, uint8_t endPage){
  if(endCol>=OLED_WIDTH||endCol<startCol||endPage>=OLED_PAGE||endPage<startPage) return 0xFF;
  OLED_CMD[0] = OLED_ADDRESS_MODE;
  OLED_CMD[1] = horizontal ? OLED_ADDR_HORIZONTAL : OLED_ADDR_VERTICAL;
  OLED_CMD[2] = OLED_ADDR_COL_RANGE;
  OLED_CMD[3] = startCol;
  OLED_CMD[4] = endCol;
  OLED_CMD[5] = OLED_ADDR_PAGE_RANGE;
  OLED_CMD[6] = startPage;
  OLED_CMD[7] = endPage;
  //Set addressing mode
  if(OLED_IIC_Write(false, OLED_CMD, 2)) return 0x01;
  //Set columns range
  if(OLED_IIC_Write(false, OLED_CMD + 2, 3)) return 0x02;
  //Set page range
  if(OLED_IIC_Write(false, OLED_CMD + 5, 3)) return 0x03;
  return 0x00;
}

/**
 * @brief This function encapsulate OLED_AddressSequence, horizontal addressing
 * @return Result of OLED_AddressSequence
 */
uint8_t OLED_AddressHorizontal(uint8_t startCol, uint8_t endCol, uint8_t startPage, uint8_t endPage){
  return OLED_AddressSequence(true, startCol, endCol, startPage, endPage);
}

/**
 * @brief This function encapsulate OLED_AddressSequence, vertical addressing
 * @return Result of OLED_AddressSequence
 */
uint8_t OLED_AddressVertical(uint8_t startCol, uint8_t endCol, uint8_t startPage, uint8_t endPage){
  return OLED_AddressSequence(false, startCol, endCol, startPage, endPage);
}

/**
 * @brief This function is composition of page addressing mode instructions
 * @param targetPage Addressing page index
 * @param startCol Addresing start column index
 * @return 0xFF means col/page out of range, 0x00 means successful, other means failure step
 */
uint8_t OLED_AddressPage(uint8_t targetPage, uint8_t startCol){
  if(targetPage>=OLED_PAGE||startCol>=OLED_WIDTH) return 0xFF;
  OLED_CMD[0] = OLED_ADDRESS_MODE;
  OLED_CMD[1] = OLED_ADDR_PAGE;
  OLED_CMD[2] = OLED_ADDR_PAGE_BASE | targetPage;
  OLED_CMD[3] = OLED_ADDR_COL_BASE_L | (startCol & 0x0F);
  OLED_CMD[4] = OLED_ADDR_COL_BASE_H | (startCol >> 4);
  if(OLED_IIC_Write(false, OLED_CMD, 2)) return 0x01;
  if(OLED_IIC_Write(false, OLED_CMD + 2, 1)) return 0x02;
  if(OLED_IIC_Write(false, OLED_CMD + 3, 2)) return 0x03;
  return 0x00;
}

/**
 * @brief 0x2E command, close all scroll instruction
 * @return Whether action succeeded
 */
uint8_t OLED_ScrollCancel(){
  OLED_CMD[0] = OLED_SCROLL_OFF;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief This function apply horizontal scroll instruction to SSD1306
 * @param right Boolean value, Scroll from left to right or right to left
 * @param speed Enum value, how many frames interval between two scroll state
 * @param startPage Scroll area begin page index
 * @param endPage Scroll area end page index
 * @return 0xFF means param error, 0x00 means successful, other meanse failure step
 */
uint8_t OLED_Scroll(uint8_t right, OLED_SCROLL_SPEED speed, uint8_t startPage, uint8_t endPage){
  if(endPage>=OLED_PAGE||endPage<startPage||speed>=OLED_SCROLL_INVALID) return 0xFF;
  if(OLED_ScrollCancel()) return 0x01;
  OLED_CMD[0] = right ? OLED_SCROLL_RIGHT : OLED_SCROLL_LEFT;
  OLED_CMD[1] = 0x00u;
  OLED_CMD[2] = startPage;
  OLED_CMD[3] = speed;
  OLED_CMD[4] = endPage;
  OLED_CMD[5] = 0x00u;
  OLED_CMD[6] = 0xFFu;
  OLED_CMD[7] = OLED_SCROLL_ON;
  //Set horizontal scroll parameters
  if(OLED_IIC_Write(false, OLED_CMD, 7)) return 0x02;
  //Start scroll process
  if(OLED_IIC_Write(false, OLED_CMD + 7, 1)) return 0x03;
  return 0x00;
}

/**
 * @brief This function apply horizontal&vertical instruction to SSD1306
 * @param right Boolean value, horizontal scroll direction, same as OLED_Scroll
 * @param down Boolean value, Scroll from up to down or down to up
 * @param hSpeed Enum value, same as OLED_Scroll funcion param speed
 * @param vSpeed 0x00~0x3F, how many rows scrolled when horizontal scroll frame switched
 * @param startPage same as OLED_Scroll
 * @param endPage same as OLED_Scroll
 * @param startRow vertical scroll area start row index
 * @param endRow vertical scroll area end row index
 * @return 0xFF means param error, 0x00 means successful, other meanse failure step
 */
uint8_t OLED_ScrollWithVertical(uint8_t right, uint8_t down, OLED_SCROLL_SPEED hSpeed, uint8_t vSpeed, uint8_t startPage, uint8_t endPage, uint8_t startRow, uint8_t endRow){
  if(endPage>=OLED_PAGE ||endPage<startPage ||hSpeed>=OLED_SCROLL_INVALID
     || endRow >= OLED_HEIGHT || endRow < startRow || vSpeed > 63 || vSpeed == 0) return 0xFF;
  if(OLED_ScrollCancel()) return 0x01;
  OLED_CMD[0] = OLED_SCROLL_RANGE_V;
  OLED_CMD[1] = startRow;
  //convert start-end value to offset-range value
  OLED_CMD[2] = endRow - startRow + 1;
  //convert vertical speed and direction to raw offset value
  OLED_CMD[3] = right ? OLED_SCROLL_RIGHT_V : OLED_SCROLL_LEFT_V;
  OLED_CMD[4] = 0x00u;
  OLED_CMD[5] = startPage;
  OLED_CMD[6] = hSpeed;
  OLED_CMD[7] = endPage;
  OLED_CMD[8] = down ? vSpeed : 64 - vSpeed;
  OLED_CMD[9] = OLED_SCROLL_ON;
  //Config vertical scroll area bound
  if(OLED_IIC_Write(false, OLED_CMD, 3)) return 0x02;
  //Config horizontal/vertical scroll command
  if(OLED_IIC_Write(false, OLED_CMD + 3, 6)) return 0x03;
  //Launch scroll process
  if(OLED_IIC_Write(false, OLED_CMD + 9, 1)) return 0x04;
  return 0x00;
}

/**
 * @brief 0xA8 command
 * @param ratio OLED panel modal rows 0x1F~0x3F
 * @return Whether configuration successful
 */
uint8_t OLED_MuxRatio(uint8_t ratio){
  if(ratio<OLED_MUX_MIN||ratio>OLED_MUX_MAX) return 0xFF;
  OLED_CMD[0] = OLED_MUX_RATIO;
  OLED_CMD[1] = ratio;
  return OLED_IIC_Write(false, OLED_CMD, 2);
}

/**
 * @brief 0x40~0x7F command
 * @param offset Offset row index, 0x00~0x3F
 * @return Whether configuration successful
 */
uint8_t OLED_OffsetGRAM(uint8_t offset){
  if(offset>=OLED_HEIGHT) return 0xFF;
  OLED_CMD[0] = OLED_RAM_OFFSET_BASE | offset;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief 0xD3 command
 * @param offset Offset row index, 0x00~0x3F
 * @return Whether configuration successful
 */
uint8_t OLED_OffsetPanel(uint8_t offset){
  if(offset>=OLED_HEIGHT) return 0xFF;
  OLED_CMD[0] = OLED_PANEL_OFFSET;
  OLED_CMD[1] = offset;
  return OLED_IIC_Write(false, OLED_CMD, 2);
}

/**
 * @brief This function is a compsition of common pins configuration
 * @param sequence Boolean value, sequence or alternative connection method
 * @param reverse Boolean value, whether reverse Left/Right common pins group
 * @param level Voltage on VCOMH, 0x00 0x20 0x30 is valid data
 * @return 0xFF means param error, 00 means successful, other means failure step
 */
uint8_t OLED_ConfigCommon(uint8_t sequence,uint8_t reverse,uint8_t level){
  if(level!=OLED_VCOMH_065&&level!=OLED_VCOMH_077&&level!=OLED_VCOMH_083) return 0xFF;
  OLED_CMD[0] = OLED_COMMON_CFG;
  OLED_CMD[1] = OLED_COMMON_BASE | ((sequence ? 0x00 : 0x01) << 4) | ((reverse ? 0x01 : 0x00) << 5);
  OLED_CMD[2] = OLED_VCOMH_SCALE;
  OLED_CMD[3] = level;
  if(OLED_IIC_Write(false, OLED_CMD, 2)) return 0x01;
  if(OLED_IIC_Write(false, OLED_CMD + 2, 2)) return 0x02;
  return 0x00;
}

/**
 * @brief 0xE3 command
 * @return Whether configuration successful
 */
uint8_t OLED_Idle(){
  OLED_CMD[0] = OLED_NOP;
  return OLED_IIC_Write(false, OLED_CMD, 1);
}

/**
 * @brief 0x8D command
 * @param enable Boolean value, open or close charge pump
 * @return Whether configuration successful
 */
uint8_t OLED_ChargePump(uint8_t enable){
  OLED_CMD[0] = OLED_CHARGE_PUMP;
  OLED_CMD[1] = enable ? 0x14 : 0x10;
  return OLED_IIC_Write(false, OLED_CMD, 2);
}

/**
 * @brief 0xD9 command
 * @param preCharge Phase 1 clock periods 0x01~0x0F
 * @param emit Phase 2 clock periods 0x01~0x0F
 * @return Whether configuration successful
 */
uint8_t OLED_ChargePhase(uint8_t preCharge, uint8_t emit){
  if(preCharge==0||preCharge>15||emit==0||emit>15) return 0xFF;
  OLED_CMD[0] = OLED_CHARGE_PHASE;
  OLED_CMD[1] = (emit << 4) | preCharge;
  return OLED_IIC_Write(false, OLED_CMD, 2);
}

/**
 * @brief 0xD5 command
 * @param osc Oscillator circuit frequency level 0x00~0x0F
 * @param divide Pre-scaler of clock circuit 0x00~0x0F
 * @return Whether configuration successful
 */
uint8_t OLED_ConfigClock(uint8_t osc,uint8_t divide){
  if(osc>15||divide>15) return 0xFF;
  OLED_CMD[0] = OLED_CLOCK_DIV;
  OLED_CMD[1] = (osc << 4) | divide;
  return OLED_IIC_Write(false, OLED_CMD, 2);
}