#ifndef OLED_DRIVER_H
#define OLED_DRIVER_H

#include <stdint.h>
#define __weak __attribute__((weak))

/// Address and register define
#define OLED_IIC_ADDR_BASE    0x78
#define OLED_CMD_REG          0x00
#define OLED_DAT_REG          0x40
/// Basic command
#define OLED_DISPLAY_ON       0xAF
#define OLED_DISPLAY_OFF      0xAE
#define OLED_COLOR_NORMAL     0xA6
#define OLED_COLOR_INVERSE    0xA7
#define OLED_DATA_RAM         0xA4
#define OLED_DATA_ALL         0xA5
#define OLED_CONTRAST         0x81
#define OLED_HORIZONTAL_LR    0xA0
#define OLED_HORIZONTAL_RL    0xA1
#define OLED_VERTICAL_UD      0xC0
#define OLED_VERTICAL_DU      0xC8
/// Addressing command
#define OLED_ADDRESS_MODE     0x20
#define OLED_ADDR_COL_RANGE   0x21
#define OLED_ADDR_PAGE_RANGE  0x22
#define OLED_ADDR_COL_BASE_L  0x00
#define OLED_ADDR_COL_BASE_H  0x10
#define OLED_ADDR_PAGE_BASE   0xB0
/// Scroll command
#define OLED_SCROLL_OFF       0x2E
#define OLED_SCROLL_ON        0x2F
#define OLED_SCROLL_RIGHT     0x26
#define OLED_SCROLL_LEFT      0x27
#define OLED_SCROLL_RIGHT_V   0x29
#define OLED_SCROLL_LEFT_V    0x2A
#define OLED_SCROLL_RANGE_V   0xA3
/// Other command
#define OLED_MUX_RATIO        0xA8
#define OLED_RAM_OFFSET_BASE  0x40
#define OLED_PANEL_OFFSET     0xD3
#define OLED_COMMON_CFG       0xDA
#define OLED_NOP              0xE3
#define OLED_VCOMH_SCALE      0xDB
#define OLED_CHARGE_PHASE     0xD9
#define OLED_CLOCK_DIV        0xD5
#define OLED_CHARGE_PUMP      0x8D
/// Fixed data or command payload
#define OLED_RESET_INTERVAL   100
#define OLED_WIDTH            128
#define OLED_HEIGHT           64    ///@attention This macro-param should determine to specific screen
#define OLED_PAGE             (OLED_HEIGHT/8)
#define OLED_ADDR_HORIZONTAL  0x00
#define OLED_ADDR_VERTICAL    0x01
#define OLED_ADDR_PAGE        0x02
#define OLED_MUX_MIN          0x0F
#define OLED_MUX_MAX          (OLED_WIDTH-1)
#define OLED_COMMON_BASE      0xCA
#define OLED_VCOMH_065        0x00
#define OLED_VCOMH_077        0x20
#define OLED_VCOMH_083        0x30
#define OLED_IIC_ADDR_ALT     0x00  ///@attention If pin DC# is pulled up, then this param should be 0x01
typedef enum{
  OLED_SCROLL_FRAME5 = 0,
  OLED_SCROLL_FRAME64,
  OLED_SCROLL_FRAME128,
  OLED_SCROLL_FRAME256,
  OLED_SCROLL_FRAME3,
  OLED_SCROLL_FRAME4,
  OLED_SCROLL_FRAME25,
  OLED_SCROLL_FRAME2,
  OLED_SCROLL_INVALID
} OLED_SCROLL_SPEED;

/// ASCII character font size
#define CHAR_SIZE_12          12
#define CHAR_SIZE_16          16
#define CHAR_SIZE_24          24

/**
 * @Note All functions below is the announcement of Command Abstract Layer.
 * These functions are all related to certain SSD1306 operation or instruction
 * and they should all implement in .c source file
 */
uint8_t OLED_Display(uint8_t on);
uint8_t OLED_ReverseColor(uint8_t reverse);
uint8_t OLED_EnableAllPanel(uint8_t all);
uint8_t OLED_Contrast(uint8_t contrast);
uint8_t OLED_ReverseHorizontal(uint8_t reverse);
uint8_t OLED_ReverseVertical(uint8_t reverse);
uint8_t OLED_AddressHorizontal(uint8_t startCol, uint8_t endCol, uint8_t startPage, uint8_t endPage);
uint8_t OLED_AddressVertical(uint8_t startCol, uint8_t endCol, uint8_t startPage, uint8_t endPage);
uint8_t OLED_AddressPage(uint8_t targetPage, uint8_t startCol);
uint8_t OLED_ScrollCancel();
uint8_t OLED_Scroll(uint8_t right, OLED_SCROLL_SPEED speed, uint8_t startPage, uint8_t endPage);
uint8_t OLED_ScrollWithVertical(uint8_t right, uint8_t down, OLED_SCROLL_SPEED hSpeed, uint8_t vSpeed, uint8_t startPage, uint8_t endPage, uint8_t startRow, uint8_t endRow);
uint8_t OLED_MuxRatio(uint8_t ratio);
uint8_t OLED_OffsetGRAM(uint8_t offset);
uint8_t OLED_OffsetPanel(uint8_t offset);
uint8_t OLED_ConfigCommon(uint8_t sequence,uint8_t reverse,uint8_t level);
uint8_t OLED_Idle();
uint8_t OLED_ChargePump(uint8_t enable);
uint8_t OLED_ChargePhase(uint8_t preCharge, uint8_t emit);
uint8_t OLED_ConfigClock(uint8_t osc,uint8_t divide);

/**
 * @Note These functions below is the announcement of Portable Hardware Layer.
 * These functions depends on sepecific platform or OLED screen, all of them is
 * implemented as "__weak" in source file, so that users can re-implement them
 * with no need to modify the source file
 */
uint8_t OLED_IIC_Write(uint8_t dc, uint8_t* payload, uint16_t size);
void OLED_Reset();
uint8_t OLED_Init();

/**
 * @Note These functions below is the graphic application method, which are not the
 * basic or core implementation for SSD1306 screen but quite important and convenient
 * during deleopment coding.
 */
uint8_t OLED_PushAll();
uint8_t OLED_Push(uint8_t page, uint8_t col, uint16_t length);
void OLED_FillAll(uint8_t data);
void OLED_Point(uint8_t x, uint8_t y, uint8_t on);
void OLED_Line(uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2, uint8_t on);
void OLED_Path(uint8_t* pathX, uint8_t* pathY, uint16_t nodes, uint8_t on, uint8_t close);
void OLED_Rectangle(uint8_t cx, uint8_t cy, float angle, uint8_t width, uint8_t height, uint8_t on);
void OLED_RegularPoly(uint8_t cx, uint8_t cy, uint8_t radius, uint8_t poly, uint8_t angle, uint8_t on);
void OLED_Arc(uint8_t cx, uint8_t cy, uint8_t radius, float startAngle, float endAngle, uint8_t on);
void OLED_Circle(uint8_t cx, uint8_t cy, uint8_t radius, uint8_t on);
void OLED_CharASCII(uint8_t x, uint8_t y, char c, uint8_t size, uint8_t on);
void OLED_StringASCII(uint8_t x,uint8_t y,const char* s,uint8_t size,uint8_t on);

#endif //OLED_DRIVER_H
